/*! Copyright(c) 2008-2015 Shenzhen TP-LINK Technologies Co.Ltd.
 *
 *\file		wps_monitor.c
 *\brief	wps monitor daemon for MTK.	
 *
 *\author	chencanjie <chencanjie@tp-link.net>
 *\version	1.0
 *\date		2015/05/04
 *
 *\history \arg	1.0, 2015/05/04, chencanjie, Create the file. 
 */

/**************************************************************************************************/
/*                                      CONFIGURATIONS                                            */
/**************************************************************************************************/

/**************************************************************************************************/
/*                                      INCLUDE_FILES                                             */
/**************************************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <net/if.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <time.h> 

#include "wps_monitor.h"
#include "wireless.h"

/**************************************************************************************************/
/* 									 VARIABLES												   */
/**************************************************************************************************/
static char wps_readbuf[WPS_READ_MAX_LEN];
static int wps_ui_sock[WLAN_BAND_NUM] = {INVALID_SOCKET};
WPS_CTX gWpsCtx[WLAN_BAND_NUM][WLAN_MAX_BSSID_NUM];
static timer_t timers[MAX_TIMER_NUM] = {0};
int wps_flag = 0x0;
int wps_dbg_level = 0;
static char uuid_2g[UUID_LEN_STR] = {0};

 /**************************************************************************************************/
 /* 									 LOCAL_FUNCTIONS										   */
 /**************************************************************************************************/
/*!
*\fn		 void _wps_monitor_printf(char* cmd, ...) 
*\brief		 print message.
*
*\param[in]	 cmd command string.
*\param[in]	 ... valist paramters.
*\return		 \
*/
void 
 _wps_monitor_printf(char* cmd, ...)
 {
	 char buf[2048] = {0};
	 char buf1[2048] ={0};
	 va_list vaList;
	 va_start(vaList, cmd);
	 vsprintf(buf, cmd, vaList);
	 va_end(vaList);
 
	 snprintf(buf1, sizeof(buf1), "echo \"%s\" > /dev/console", buf);
	 system(buf1);
 }

/*!
*\fn		_wps_monitor_daemonize() 
*\brief		 daemonize this process.
*/
static void 
_wps_monitor_daemonize() 
{
	unsigned int  fd = 0;

	if (fork() != 0) 
		exit(0);	 /* parent exits */

	setsid();		 /* create a new session */

	/* Every output goes to /dev/null. If Redis is daemonized but
	* the 'logfile' is set to 'stdout' in the configuration file
	* it will not log at all. */
	if ((fd = open("/dev/null", O_RDWR, 0)) != -1) 
	{
		dup2(fd, STDIN_FILENO);
		dup2(fd, STDOUT_FILENO);
		dup2(fd, STDERR_FILENO);

		if (fd > STDERR_FILENO) 
		{
			close(fd);
		}
	}
}

/*!
*\fn			 open_udp_socket(char *addr, int port)
*\brief		 open a udp socket.
*
*\param[in]	 addr	 specify the socket address .
*\param[in]	 port	 specify the port.
*
*\return		 return socket fd if succeed, otherwise return ERROR.
*/
static int 
_open_udp_socket(char *addr, int port)
{
	int reuse = 1;
	int sock_fd;
	struct sockaddr_in sockaddr;

	/* open loopback socket to communicate with worker */
	memset(&sockaddr, 0, sizeof(sockaddr));
	sockaddr.sin_family = AF_INET;
	sockaddr.sin_addr.s_addr = inet_addr(addr);
	sockaddr.sin_port = htons(port);

	if ((sock_fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) 
	{
		wpsDbg("Unable to create loopback socket");
		goto exit0;
	}

	if (setsockopt(sock_fd, SOL_SOCKET, SO_REUSEADDR, (char*)&reuse, sizeof(reuse)) < 0) {
		wpsDbg("Unable to setsockopt to loopback socket %d.", sock_fd);
		goto exit1;
	}

	if (bind(sock_fd, (struct sockaddr *)&sockaddr, sizeof(sockaddr)) < 0) 
	{
		wpsDbg("Unable to bind to loopback socket %d", sock_fd);
		goto exit1;
	}

	wpsDbg("opened loopback socket %d in port %d", sock_fd, port);
	return sock_fd;

/* error handling */
exit1:
	close(sock_fd);

exit0:
	wpsDbg("failed to open loopback socket");
	return ERROR;
}
 
/*!
*\fn			 _close_udp_socket(int fd)
*\brief		 close socket.
*
*\param[in]	 fd  specify the socket fd.
*/
static inline 
void _close_udp_socket(int fd)
{
	if (fd >= 0)
		close(fd);
}

/*!
*\fn			_is_Int_Between(int val, int range1, int range2)
*\brief		 check value between range1 and range2 or not
*
*\param[in]	 val   	 specify the check value.
*\param[in]	 range1	 specify the start range value.
*\param[in]	 range2	 specify the end range value.
*
*\return		 return TRUE if between ranges, otherwise return FALSE.
*/
static BOOL
_is_Int_Between(int val, int range1, int range2)
{
	int max = (range1 > range2)? range1:range2;
	int min = (range1 < range2)? range1:range2;

	if (val >= min && val <= max)
		return TRUE;
	else
		return FALSE;
}

/*!
*\fn			_tpCreateTimer(int signum, sighandler handler_func)
*\brief		 create a timer, when time's up, will send a signal "signum" to the handler_func
*
*\param[in]	 signum   	 	specify the signal number.
*\param[in]	 handler_func	specify the handler function.
*
*\return		 return the tid.
*/
static timer_t 
_tpCreateTimer(int signum, sighandler handler_func)
{
	timer_t tid;
	struct sigevent se;
	int index = signum - SIGNUM_MIN;

	if (!_is_Int_Between(signum, SIGNUM_MIN, SIGNUM_MAX))
	{
		wpsError("illegal signum");
		return ERROR;
	}

	/* signal already registered? */
	if (timers[index])
	{
		wpsError("signal(%d) has been used", signum);
		return ERROR;
	}

	memset(&se, 0, sizeof(se));

	signal(signum, handler_func);

	se.sigev_notify = SIGEV_SIGNAL;
	se.sigev_signo = signum;
	se.sigev_notify_function = handler_func;

	if (timer_create(CLOCK_MONOTONIC, &se, &tid) < 0)
	{
		wpsError("timer_create error");
		return ERROR;
	}

	timers[index] = tid;

	return tid;
}

/*!
*\fn			_tpSetTimerNs(timer_t timer_id, long firstRun, long sec, long nsec)
*\brief		setting timmer.
*
*\param[in]	 timer_id   	specify the tid.
*\param[in]	 firstRun		specify the first run second.
*\param[in]	 sec			specify the interval second.
*\param[in]	 nsec		specify the interval nsecond.
*
*\return		 OK/ERROR.
*/
static int 
_tpSetTimerNs(timer_t timer_id, long firstRun, long sec, long nsec)
{
	struct itimerspec ts;
	struct itimerspec ots;

	ts.it_value.tv_sec = firstRun;
	ts.it_value.tv_nsec = 0;
	ts.it_interval.tv_sec = sec;
	ts.it_interval.tv_nsec = nsec;

	if (timer_settime(timer_id, 0, &ts, &ots) < 0)
	{
		wpsError("timer_settime error");
		return ERROR;
	}

	return OK;
}

/*!
*\fn			int _tpSetTimer(timer_t timer_id, long firstRun, long interval)
*\brief		let the timer run.
*
*\param[in]	 timer_id   	specify the tid.
*\param[in]	 firstRun		specify the first run second.
*\param[in]	 interval		specify the interval second.
*
*\return		 OK/ERROR.
*/
static int 
_tpSetTimer(timer_t timer_id, long firstRun, long interval)
{
	return _tpSetTimerNs(timer_id, firstRun, interval, 0);
}

/*!
*\fn			_tpDeleteTimer(timer_t timer_id)
*\brief		delete timmer.
*
*\param[in]	 timer_id   	specify the tid.
*
*\return		 OK/ERROR.
*/
static int 
_tpDeleteTimer(timer_t timer_id)
{
	int i;

	for (i = 0; i < MAX_TIMER_NUM; i ++)
	{
		if (timers[i] == timer_id)
		{
			timers[i] = 0;
			return timer_delete(timer_id);
		}
	}

	wpsDbg("delete timer id %d failed: not found\n", timer_id);
	return ERROR;
}

/*!
*\fn			_tpDownTimer(timer_t timer_id)
*\brief		stop timmer.
*
*\param[in]	 timer_id   	specify the tid.
*/
static void 
_tpDownTimer(timer_t timer_id)
{
	if (timer_id > 0)
		_tpSetTimer(timer_id, 0, 0);
}

/*!
*\fn			_wps_set_monitor_data(WLAN_BAND band, int wscStatus, MONITOR_DATA *pdata)
*\brief		set the data sended to wpsd.
*
*\param[in]	 band   		specify the band.
*\param[in]	 wscStatus   	specify the wps status.
*\param[in]	 pdata   		specify the pointer of the data sended to wpsd.
*/
static void 
_wps_set_monitor_data(WLAN_BAND band, int wscStatus, MONITOR_DATA *pdata)
{
	if(!pdata)
		return;
	
	pdata->band = band;
	pdata->finished = TRUE;
	pdata->status = wscStatus;
}

/*!
*\fn			_sendMsg_to_wpsd(MONITOR_DATA *pdata)
*\brief		send message to wpsd to report wps status.
*
*\param[in]	 pdata   	specify the pointer of the data sended to wpsd.
*
*\return		 OK/ERROR.
*/
static int 
_sendMsg_to_wpsd(MONITOR_DATA *pdata)
{
	int wpsd_fd = -1;
	struct sockaddr_in to;
	int sentBytes = 0;

	if ((wpsd_fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		wpsError("create socket failed\n");
		goto exit;
	}
 
	/* send to wpsd */
	to.sin_addr.s_addr = inet_addr(WPS_LOOPBACK_ADDR);
	to.sin_family = AF_INET;
	if (WLAN_BAND_2G == pdata->band)
	{
		wpsDbg("use WPSD_WL0_MONITOR_PORT.\n");
		to.sin_port = htons(WPSD_WL0_MONITOR_PORT);
	}
	else
	{
		wpsDbg("use WPSD_WL1_MONITOR_PORT.\n");
		to.sin_port = htons(WPSD_WL1_MONITOR_PORT);
	}
 
	sentBytes = sendto(wpsd_fd, pdata, sizeof(MONITOR_DATA), 0, (struct sockaddr *)&to, sizeof(struct sockaddr_in));

	if (sentBytes != sizeof(MONITOR_DATA)) {
		wpsError("send buf to wpsd failed, sentBytes = %d, sizeof(MONITOR_DATA) = %d\n", sentBytes, sizeof(MONITOR_DATA));
		goto exit;
	}

	 /* Sleep 100 ms to make sure
		 wpsd have received socket */
	usleep(100*1000);

	close(wpsd_fd);

	return OK;
	
exit:	
	if (wpsd_fd >= 0)
		 close(wpsd_fd);

	 /* Show error message ?  */
	 return ERROR;
}

void wlanWpsCtxInit(int band, int vapIdx)
{
	WPS_CTX *wpsCtx = &gWpsCtx[band][vapIdx];
	memset(wpsCtx, 0, sizeof(WPS_CTX));
		
	wpsCtx->status = WPS_IDLE;
	wpsCtx->role = WPS_REGISTRAR;
	wpsCtx->apConfigured = FALSE;

	wpsCtx->tid = 0;
	wpsCtx->timerBusy = FALSE;

	wpsCtx->wscTimeoutCounter = 0;
	wpsCtx->wscBusy = FALSE;

	wpsCtx->pid = 0;
	pthread_mutex_init(&wpsCtx->mutex, NULL);
}

/*!
*\fn			_wlan_iwpriv_set_Int(const char *ifname, const char *cmd, int value)
*\brief		send iwpriv command.
*
*\param[in]	 ifname   	specify the interface name.
*\param[in]	 cmd	   	specify the command name.
*\param[in]	 value	   	specify the command value.
*/
static void 
_wlan_iwpriv_set_Int(const char *ifname, const char *cmd, int value)
{
	char cmdline[256] = {0};
	sprintf(cmdline, "iwpriv %s set %s=%d", ifname, cmd, value);
	wpsDbg("%s", cmdline);
	system(cmdline);
}

static void 
_wlan_iwpriv_set_Str(const char *ifname, const char *cmd, const char *value)
{
	char cmdline[256] = {0};
	sprintf(cmdline, "iwpriv %s set %s=%s", ifname, cmd, value);
	wpsDbg("%s", cmdline);
	system(cmdline);
}

/*!
*\fn		_wlan_sta_vap_name(char *vapName, int band)
*\brief		 convert input index to vapName.
*
*\param[in]	 vapName   vap name.
*\param[in]	 band         vap band.
*/
static void 
_wlan_sta_vap_name(char *vapName, int band)
{
	int vapIdx;

	if (!vapName)
		return;

	if (WLAN_BAND_2G == band)
	{
		vapIdx = WLAN_2G_STA_IDX;
		sprintf(vapName, WLAN_STA_2G_IFNAME_FORMAT, vapIdx);
	}
	else
	{
		vapIdx = WLAN_5G_STA_IDX;
		sprintf(vapName, WLAN_STA_5G_IFNAME_FORMAT, vapIdx);
	}
}

void _wlan_ap_vap_name(char *vapName, int band)
{
	int vapIdx;
	
	if (!vapName)
		return;

	if (WLAN_BAND_2G == band)
	{
		vapIdx = WLAN_2G_AP_IDX;
		sprintf(vapName, WLAN_AP_2G_IFNAME_FORMAT, vapIdx);
	}
	else
	{
		vapIdx = WLAN_5G_AP_IDX;
		sprintf(vapName, WLAN_AP_5G_IFNAME_FORMAT, vapIdx);
	}
}

/*!
*\fn			_wlan_Wps_Get_CtxRole(int band, int vapIdx)
*\brief		get wps role
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*
*\return		 	WPS_ENROLLEE/WPS_REGISTRAR
*/
static int 
_wlan_Wps_Get_CtxRole(int band, int vapIdx)
{
	return gWpsCtx[band][vapIdx].role;
}

/*!
*\fn			_wlan_Wps_Check_CtxRole(WPS_ROLE role, int band, int vapIdx)
*\brief		set wps role
*
*\param[in]	 role   	specify the wps role.
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_Set_CtxRole(WPS_ROLE role, int band, int vapIdx)
{
	gWpsCtx[band][vapIdx].role = role;
}

/*!
*\fn			_wlan_Wps_Get_TimerId(int band, int vapIdx)
*\brief		get wps tid
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*
*\return		 	tid
*/
static timer_t 
_wlan_Wps_Get_TimerId(int band, int vapIdx)
{
	return gWpsCtx[band][vapIdx].tid;
}

/*!
*\fn			_wlan_Wps_Set_TimerId(timer_t tid, int band, int vapIdx)
*\brief		set wps tid
*
*\param[in]	 tid   	specify the wps tid.
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_Set_TimerId(timer_t tid, int band, int vapIdx)
{
	gWpsCtx[band][vapIdx].tid = tid;
}

/*!
*\fn			_wlan_Wps_TimerInProgress(int band, int vapIdx)
*\brief		indicate wps is in progess or not
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*
*\return		 	TRUE/FALSE
*/
static BOOL 
_wlan_Wps_TimerInProgress(int band, int vapIdx)
{
	return gWpsCtx[band][vapIdx].timerBusy;
}

/*!
*\fn			_wlan_Wps_Set_TimerInProgress(BOOL state, int band, int vapIdx)
*\brief		set wps is in progess or not
*
*\param[in]	 state   	specify the wps state.
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_Set_TimerInProgress(BOOL state, int band, int vapIdx)
{
	gWpsCtx[band][vapIdx].timerBusy = state;
}

/*!
*\fn			_wlan_Wps_WscInProgress(int band, int vapIdx)
*\brief		indicate wps is start or not
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*
*\return		 	TRUE/FALSE
*/
static BOOL 
_wlan_Wps_WscInProgress(int band, int vapIdx)
{
	return gWpsCtx[band][vapIdx].wscBusy;
}

/*!
*\fn			_wlan_Wps_Set_WscInProgress(int state, int band, int vapIdx)
*\brief		set wps is start or not
*
*\param[in]	 state   	specify the wps state.
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_Set_WscInProgress(int state, int band, int vapIdx)
{
	gWpsCtx[band][vapIdx].wscBusy = state;
}

/*!
*\fn			_wlan_Wps_WscTimeout(int band, int vapIdx)
*\brief		indicate wps is timeout or not
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*
*\return		 	TRUE/FALSE
*/
static BOOL 
_wlan_Wps_WscTimeout(int band, int vapIdx)
{
	return (gWpsCtx[band][vapIdx].wscTimeoutCounter >= WPS_TIMEOUT_TIMES  ); /*WPS_AP_TIMEOUT_SECS); */
}

/*!
*\fn			_wlan_Wps_WscTimeout_Reset(int band, int vapIdx)
*\brief		reset timeout value
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_WscTimeout_Reset(int band, int vapIdx)
{
	gWpsCtx[band][vapIdx].wscTimeoutCounter = 0;
}

/*!
*\fn			_wlan_Wps_WscTimeout_Lapse(int band, int vapIdx)
*\brief		increase timeout value
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_WscTimeout_Lapse(int band, int vapIdx)
{
	int lapse = 1; 	/*	WPS_AP_CATCH_CONFIGURED_TIMER / MS;	*/
	gWpsCtx[band][vapIdx].wscTimeoutCounter += lapse;
}

/*!
*\fn			_wlan_Wps_Reset_Timer(int band, int vapIdx)
*\brief		reset wps timer
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_Reset_Timer(int band, int vapIdx)
{
	int tid = _wlan_Wps_Get_TimerId(band, vapIdx);
	if (tid > 0)
	{
		_tpDownTimer(tid);

		if (OK == _tpDeleteTimer(tid))
			_wlan_Wps_Set_TimerId(0, band, vapIdx);
	}

	_wlan_Wps_Set_WscInProgress(FALSE, band, vapIdx);
	_wlan_Wps_Set_TimerInProgress(FALSE, band, vapIdx);
}

static void
_wlan_Pin_Background_Wps_Reset_Timer(int band, int vapIdx)
{
	int tid = _wlan_Wps_Get_TimerId(band, vapIdx+1);
	if (tid > 0)
	{
		_tpDownTimer(tid);

		if (OK == _tpDeleteTimer(tid))
			_wlan_Wps_Set_TimerId(0, band, vapIdx+1);
	}

	_wlan_Wps_Set_WscInProgress(FALSE, band, vapIdx+1);
	_wlan_Wps_Set_TimerInProgress(FALSE, band, vapIdx+1);
}

/*!
*\fn			_wlan_Wps_Signum2Band(int signum)
*\brief		check the signal is legal or not
*
*\param[in]	 signum   	specify the signal number.
*
*\return		 	OK/ERROR
*/
static int 
_wlan_Wps_Signum2Band(int signum)
{
	if (_is_Int_Between(signum, SIGNUM_WPS_2G, SIGNUM_WPS_2G_4))
		return WLAN_BAND_2G;
	else if (_is_Int_Between(signum, SIGNUM_WPS_5G, SIGNUM_WPS_5G_4))
		return WLAN_BAND_5G;
	else
		return ERROR;
}

/*!
*\fn			_wlan_Wps_Signum2ApIdx(int signum)
*\brief		change signal to vap index
*
*\param[in]	 signum   	specify the signal number.
*
*\return		 	vap index
*/
static int 
_wlan_Wps_Signum2ApIdx(int signum)
{
	int band = _wlan_Wps_Signum2Band(signum);
	int ret = ERROR;

	if (band != ERROR) {
		if (band == WLAN_BAND_2G)
			ret = signum - SIGNUM_WPS_2G;
		else
			ret = signum - SIGNUM_WPS_5G;
	}

	return ret;
}

/*!
*\fn			_wlan_Wps_Get_WscStatus(char *ifname)
*\brief		ioctl to get wps status
*
*\param[in]	 ifname   	specify the net interface name.
*
*\return		 	wps status
*/
static int 
_wlan_Wps_Get_WscStatus(char *ifname)
{
	int sock = -1;
	struct iwreq wrq;
	int data = STATUS_WSC_IDLE;

	if (sock < 0)
		sock = socket(AF_INET, SOCK_DGRAM, 0);

	if (sock < 0) {
		wpsError("socket error.");
		goto exit;
	}

	strcpy(wrq.ifr_name, ifname);
	wrq.u.data.length = sizeof(data);
	wrq.u.data.pointer = (caddr_t) &data;
	wrq.u.data.flags = RT_OID_WSC_QUERY_STATUS;

	if (ioctl(sock, RT_PRIV_IOCTL, &wrq) < 0){
		//wpsError("get wsc status error.");
	}

	close(sock);

	return data;

exit:
	if (sock >= 0)
		 close(sock);

	return ERROR;
}

/*!
*\fn			_wlan_Wps_Signum(int band, int apIdx)
*\brief		get signal number
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*
*\return		 	signal number
*/
static int 
_wlan_Wps_Signum(int band, int apIdx)
{
	if (!_is_Int_Between(band, WLAN_BAND_MIN, WLAN_BAND_MAX) ||
			!_is_Int_Between(apIdx, 0, WLAN_MAX_BSSID_NUM))
		return ERROR;

	if (WLAN_BAND_2G == band)
		return (SIGNUM_WPS_2G + apIdx);
	else
		return (SIGNUM_WPS_5G + apIdx);
}

static void 
_wlan_Wps_Pin_Background_Session(int signum, WLAN_BAND band )
{
	int WscStatus = 0;
	int apIdx = 1;
	char vapName[IFNAMESIZ] = {0};
	MONITOR_DATA data;
	data.isFromPinBackground = TRUE;

	if (!_is_Int_Between(signum, SIGNUM_WPS_2G, SIGNUM_WPS_5G_4)) {
		wpsError("Illegal signum:%d", signum);
		return;
	}

	band = _wlan_Wps_Signum2Band(signum);
	apIdx = _wlan_Wps_Signum2ApIdx(signum);
	if (ERROR == band || ERROR == apIdx) {
		wpsError("Illegal parameter band & apIdx:%d, %d", band, apIdx);
		return;
	}

	_wlan_ap_vap_name(vapName, band);

	WscStatus = _wlan_Wps_Get_WscStatus(vapName);
	//wpsInfo("====>>>> WscStatus=%d[%s]", WscStatus, vapName);
	if (WscStatus == STATUS_WSC_CONFIGURED)
	{
		wpsInfo("PIN background success");
		_wps_set_monitor_data(band, WscStatus, &data);
		_sendMsg_to_wpsd(&data);
		return;
	}
}

static void
_wlan_Wps_Pin_Background_Session_2G(int signum)
{
	_wlan_Wps_Pin_Background_Session(signum, WLAN_BAND_2G );
}

static void
_wlan_Wps_Pin_Background_Session_5G(int signum)
{
	_wlan_Wps_Pin_Background_Session(signum, WLAN_BAND_5G );
}

static void 
_wlan_Wps_Pin_Background_NewSession(int band, int apIdx, int delay)
{
	timer_t tid = _wlan_Wps_Get_TimerId(band, apIdx+1);

	if (tid <= 0) {
		int signum = _wlan_Wps_Signum(band, apIdx+1);
		if ( band == WLAN_BAND_2G )
		{
			tid = _tpCreateTimer(signum, _wlan_Wps_Pin_Background_Session_2G);
		}
		else if ( band == WLAN_BAND_5G )
		{
			tid = _tpCreateTimer(signum, _wlan_Wps_Pin_Background_Session_5G);
		}
		_wlan_Wps_Set_TimerId(tid, band, apIdx+1);
		wpsDbg("WPS timer ID:%d", tid);
	}

	if (tid <= 0) {
		wpsError("WPS timer create error");
		return;
	}

	_wlan_Wps_Set_WscInProgress(TRUE, band, apIdx+1);
	_tpSetTimerNs(tid, delay, 2, 0);/*check per 2 second*/
}

/*!
*\fn			_wlan_Wps_Signum(int band, int apIdx)
*\brief		check wps status main handler, 1 times per second.
*
*\param[in]	 signum 	specify the signal number.
*\param[in]	 band   	specify the wps band.
*/
static void 
_wlan_Wps_Session(int signum, WLAN_BAND band )
{
	int WscStatus = 0;
	int apIdx = 0;
	char vapName[IFNAMESIZ] = {0};
	MONITOR_DATA data;
	data.isFromPinBackground = FALSE;

	if (!_is_Int_Between(signum, SIGNUM_WPS_2G, SIGNUM_WPS_5G_4)) {
		wpsError("Illegal signum:%d", signum);
		return;
	}

	band = _wlan_Wps_Signum2Band(signum);
	apIdx = _wlan_Wps_Signum2ApIdx(signum);
	if (ERROR == band || ERROR == apIdx) {
		wpsError("Illegal parameter band & apIdx:%d, %d", band, apIdx);
		return;
	}

	if(WPS_ENROLLEE == _wlan_Wps_Get_CtxRole(band, apIdx))
		_wlan_sta_vap_name(vapName, band);
	else
		_wlan_ap_vap_name(vapName, band);

	WscStatus = _wlan_Wps_Get_WscStatus(vapName);
	wpsInfo("WscStatus=%d[%d], wscTimeoutCounter=%d[%d]", WscStatus, band, gWpsCtx[band][apIdx].wscTimeoutCounter, band);
	if (_is_Int_Between(WscStatus, STATUS_WSC_LINK_UP, STATUS_WSC_EAP_M8_SENT) &&
			!_wlan_Wps_TimerInProgress(band, apIdx))
	{
		wpsDbg("Start to monitor WSC Status...");
		_wlan_Wps_Set_TimerInProgress(TRUE, band, apIdx);
		//_wlan_Wps_WscTimeout_Reset(band, apIdx);
	}

	_wlan_Wps_WscTimeout_Lapse(band, apIdx);

	if (_wlan_Wps_WscTimeout(band, apIdx))
	{
		wpsDbg("WSC failed, Timeout");
		/* Timeout, WSC process failed. */
		_wlan_Wps_WscTimeout_Reset(band, apIdx);
		_wlan_Wps_Reset_Timer(band, apIdx);
		_wps_set_monitor_data(band, STATUS_WSC_FAIL, &data);
		_sendMsg_to_wpsd(&data);
		_wlan_Wps_Pin_Background_NewSession(band, apIdx, 1);
		return;
	}
	
	if (WscStatus == STATUS_WSC_NOTUSED)
	{
		wpsDbg("WSC cancel");
		_wlan_Wps_WscTimeout_Reset(band, apIdx);
		_wlan_Wps_Reset_Timer(band, apIdx);
		_wps_set_monitor_data(band, WscStatus, &data);
		_sendMsg_to_wpsd(&data);
		_wlan_Wps_Pin_Background_NewSession(band, apIdx, 1);
		return;
	}

	/* still in progress and keep monitoring. */
	if (WscStatus == STATUS_WSC_CONFIGURED)
	{
		wpsDbg("WSC success");
		_wlan_Wps_WscTimeout_Reset(band, apIdx);
		_wlan_Wps_Reset_Timer(band, apIdx);
		_wps_set_monitor_data(band, WscStatus, &data);
		_sendMsg_to_wpsd(&data);
		_wlan_Wps_Pin_Background_NewSession(band, apIdx, WPS_TIMEOUT_TIMES-gWpsCtx[band][apIdx].wscTimeoutCounter);
		return;
	}

	/* Driver 1.9 supports AP PBC Session Overlapping Detection. */
	if (WscStatus == STATUS_WSC_PBC_SESSION_OVERLAP)
	{
		wpsDbg("WSC overlap");
		_wlan_Wps_WscTimeout_Reset(band, apIdx);
		_wlan_Wps_Reset_Timer(band, apIdx);
		_wps_set_monitor_data(band, WscStatus, &data);
		_sendMsg_to_wpsd(&data);
		_wlan_Wps_Pin_Background_NewSession(band, apIdx, 1);
		return;
	}

}

/*!
*\fn			_wlan_Wps_Session_2G(int signum)
*\brief		start 2G wps handler
*
*\param[in]	 vapIdx  	specify the signal number.
*/
static void 
_wlan_Wps_Session_2G(int signum)
{
	_wlan_Wps_Session(signum, WLAN_BAND_2G );
}

/*!
*\fn			_wlan_Wps_Session_5G(int signum)
*\brief		start 5G wps handler
*
*\param[in]	 vapIdx  	specify the signal number.
*/
static void 
_wlan_Wps_Session_5G(int signum)
{
	_wlan_Wps_Session(signum, WLAN_BAND_5G );
}

/*!
*\fn			_wlan_Wps_NewSession(int band, int apIdx)
*\brief		start wps handler
*
*\param[in]	 band   	specify the wps band.
*\param[in]	 vapIdx  	specify the wps vap index.
*/
static void 
_wlan_Wps_NewSession(int band, int apIdx)
{
	timer_t tid = _wlan_Wps_Get_TimerId(band, apIdx);

	if (tid <= 0) {
		int signum = _wlan_Wps_Signum(band, apIdx);
		if ( band == WLAN_BAND_2G )
		{
			tid = _tpCreateTimer(signum, _wlan_Wps_Session_2G);
		}
		else if ( band == WLAN_BAND_5G )
		{
			tid = _tpCreateTimer(signum, _wlan_Wps_Session_5G);
		}
		_wlan_Wps_Set_TimerId(tid, band, apIdx);
		wpsDbg("WPS timer ID:%d", tid);
	}

	if (tid <= 0) {
		wpsError("WPS timer create error");
		return;
	}

	_wlan_Wps_Set_WscInProgress(TRUE, band, apIdx);
	_tpSetTimerNs(tid, 1, 1, 0);
}

/*!
*\fn			_wps_handle_signal(int sig)
*\brief		signal handle function.
*
*\param[in]	 sig		specify the signal number.
*/
static void 
_wps_handle_signal(int sig)
{
    if (sig == SIGUSR1)
    {
        wps_dbg_level++;
		wpsInfo("wps_dbg_level: %d", wps_dbg_level);
    }
    else if (sig == SIGUSR2)
    {
        wps_dbg_level--;
		wpsInfo("wps_dbg_level: %d", wps_dbg_level);
    }
	else if (sig == SIGTERM)
	{
		wps_flag |= WPS_FLAG_SHUTDOWN;
	}
}

/*!
*\fn			_wps_register_signal(void)
*\brief		regist wps_monitor signal to kill process or change debug level.
*
*\return		OK/ERROR
*/
static int 
_wps_register_signal(void)
{
    struct sigaction sa;
    
    memset(&sa, 0, sizeof(sa));

    sa.sa_handler = _wps_handle_signal;
    
    if (sigaction(SIGUSR1, &sa, NULL) < 0)
    {
        wpsError("SIGUSR1 err: %s", strerror(errno));
        return ERROR;
    }

    if (sigaction(SIGUSR2, &sa, NULL) < 0)
    {
        wpsError("SIGUSR1 err: %s", strerror(errno));
        return ERROR;
    }

	if (sigaction(SIGTERM, &sa, NULL) < 0)
    {
        wpsError("SIGTERM err: %s", strerror(errno));
        return ERROR;
    }

    return OK;
}

/*!
*\fn		_wps_monitor_open()
*\brief		 open wps_monitor daemon workspace.
*
*\return		 OK/ERROR
*/
static int _wps_monitor_open()
{
	int band = 0;

	for (band = 0; band < WLAN_BAND_NUM; band++)
	{
		wlanWpsCtxInit(band, WLAN_STA_IDX);
		
		if(WLAN_BAND_2G == band)
			wps_ui_sock[band] = _open_udp_socket(WPS_LOOPBACK_ADDR, WL0_WPS_UI_PORT);
		else 
			wps_ui_sock[band] = _open_udp_socket(WPS_LOOPBACK_ADDR, WL1_WPS_UI_PORT);

		if(wps_ui_sock[band] == ERROR)
		{
			wpsError("udp socket init failed");
			return ERROR;
		}
	}
	_wlan_Wps_Pin_Background_NewSession(WLAN_BAND_2G, WLAN_AP_IDX, 1);
	_wlan_Wps_Pin_Background_NewSession(WLAN_BAND_5G, WLAN_AP_IDX, 1);	

	return OK;
}

/*!
 *\fn	   _wps_repeater_start(OP_WIFI_BAND_TYPE band)
 *\brief		start wps process in repeater mode.
 *
 *\return		OK/ERROR
 */
static int 
_wps_repeater_start(WLAN_BAND band)
{
	char vapName[IFNAMESIZ];
	
	if (_wlan_Wps_WscInProgress(band, WLAN_STA_IDX))
	{
		wpsError("wsc is busy, can not start wps process.");
		return ERROR;
	}
	
	_wlan_Wps_Set_CtxRole(WPS_ENROLLEE, band, WLAN_STA_IDX);
	
	_wlan_sta_vap_name(vapName, band);
	_wlan_iwpriv_set_Int(vapName, "ApCliEnable", 1);
	_wlan_iwpriv_set_Int(vapName, "WscConfMode", 1);
	//_wlan_iwpriv_set_Int(vapName, "WscConfStatus", 1);
	_wlan_iwpriv_set_Int(vapName, "WscMode", 2);
	_wlan_iwpriv_set_Int(vapName, "WscGetConf", 1);
	//_wlan_iwpriv_set_Int(vapName, "ApCliEnable", 1);

	_wlan_Wps_NewSession(band, WLAN_STA_IDX);

	return OK;
}

static int
_wps_repeater_stop(WLAN_BAND band)
{
	char vapName[IFNAMESIZ];
	if (FALSE == _wlan_Wps_WscInProgress(band, WLAN_STA_IDX))
	{
		wpsError("wsc is no in progress, can not stop wps process.");
		return ERROR;
	}

	_wlan_Wps_Set_CtxRole(WPS_ENROLLEE, band, WLAN_STA_IDX);
	
	_wlan_sta_vap_name(vapName, band);
	_wlan_iwpriv_set_Int(vapName, "WscStop", 1);
}

static int 
_wps_get_UUID(char * uuid, char *ifName)
{
	struct iwreq	iwr;
	int 		s = 0;
	char 		*buf = uuid;
	int	 		len;

	if ( s == 0 )
	{
		s = socket(AF_INET, SOCK_DGRAM, 0);
		if (s <= 0)
		{
			wpsError("socket(SOCK_DRAGM)");
			goto exit;
		}
	}
	memset(&iwr, 0, sizeof(struct iwreq) );
	sprintf( iwr.ifr_name, "%s", ifName );
	
	iwr.u.data.pointer = (void *)buf;
	iwr.u.data.length = sizeof(buf);
	iwr.u.data.flags = RT_OID_WSC_UUID;
	if ( ioctl( s, RT_PRIV_IOCTL, &iwr) < 0 )
	{
		wpsError("func:%s(%d),unable to get wsc profile information", __func__, __LINE__ );
		goto exit;
	}
	
	len = iwr.u.data.length ;
	if( len < 0 )
		goto exit;

	close(s);

	return 0;

exit:
	if (s >= 0)
		 close(s);

	return -1;
}

static void 
_wps_UUID_str2e(char * uuid)
{
	int index = 0;
	char uuid_e[UUID_LEN_STR];
	int index_e = 0;
	for(index = 0; index < UUID_LEN_STR; index++)
	{
		if('-' != uuid[index])
		{
			uuid_e[index_e] = uuid[index];
			index_e++;
		}	
	}
	strncpy(uuid, uuid_e, UUID_LEN_STR);
}

static void 
_wlan_change_UUID_5g(band)
{
	char vapName[IFNAMESIZ];
	_wlan_ap_vap_name(vapName, band);

	if(band == WLAN_BAND_5G && 0 != strlen(uuid_2g))
	{
		_wlan_iwpriv_set_Str(vapName, "WscUUID_E", uuid_2g);
	}
}

static void 
_wps_backup_UUID_2g(band)
{
	char vapName[IFNAMESIZ];
	if(band == WLAN_BAND_2G)
	{
		_wlan_ap_vap_name(vapName, band);
		_wps_get_UUID(uuid_2g, vapName);
		_wps_UUID_str2e(uuid_2g);
	}
}

static int 
_wps_ap_pbc_start(WLAN_BAND band, BOOL isApConfig)
{
	char vapName[IFNAMESIZ];

	if (_wlan_Wps_WscInProgress(band, WLAN_AP_IDX))
	{
		wpsError("wsc is busy, can not start wps process.");
		return ERROR;
	}
	_wlan_Pin_Background_Wps_Reset_Timer(band, WLAN_AP_IDX);

	_wlan_Wps_Set_CtxRole(WPS_REGISTRAR, band, WLAN_AP_IDX);
	
	_wlan_ap_vap_name(vapName, band);

	_wlan_change_UUID_5g(band);
	
	_wlan_iwpriv_set_Int(vapName, "WscConfMode", 7);
	if(isApConfig)
		_wlan_iwpriv_set_Int(vapName, "WscConfStatus", 2);
	else
		_wlan_iwpriv_set_Int(vapName, "WscConfStatus", 1);
	_wlan_iwpriv_set_Int(vapName, "WscMode", 2);
	_wlan_iwpriv_set_Int(vapName, "WscGetConf", 1);

	_wlan_Wps_NewSession(band, WLAN_AP_IDX);
}

static int
_wps_ap_pin_start(WLAN_BAND band, unsigned char* pinCode, BOOL isApConfig)
{
	char vapName[IFNAMESIZ];
	char pin[9] = {0};

	if (_wlan_Wps_WscInProgress(band, WLAN_AP_IDX))
	{
		wpsError("wsc is busy, can not start wps process.");
		return ERROR;
	}
	_wlan_Pin_Background_Wps_Reset_Timer(band, WLAN_AP_IDX);

	_wlan_Wps_Set_CtxRole(WPS_REGISTRAR, band, WLAN_AP_IDX);
	strncpy(pin, pinCode, 8);

	_wlan_ap_vap_name(vapName, band);

	_wlan_change_UUID_5g(band);
	
	_wlan_iwpriv_set_Int(vapName, "WscConfMode", 7);
	if(isApConfig)
		_wlan_iwpriv_set_Int(vapName, "WscConfStatus", 2);
	else
		_wlan_iwpriv_set_Int(vapName, "WscConfStatus", 1);
	_wlan_iwpriv_set_Int(vapName, "WscMode", 1);
	_wlan_iwpriv_set_Str(vapName, "WscPinCode", pin);
	_wlan_iwpriv_set_Int(vapName, "WscGetConf", 1);

	_wlan_Wps_NewSession(band, WLAN_AP_IDX);
}

static int
_wps_ap_stop(WLAN_BAND band)
{
	char vapName[IFNAMESIZ];
	if (FALSE == _wlan_Wps_WscInProgress(band, WLAN_STA_IDX))
	{
		wpsError("wsc is no in progress, can not stop wps process.");
		return ERROR;
	}

	_wlan_Wps_Set_CtxRole(WPS_REGISTRAR, band, WLAN_AP_IDX);
	
	_wlan_ap_vap_name(vapName, band);
	_wlan_iwpriv_set_Int(vapName, "WscStop", 1);
}

/*!
*\fn		_wps_monitor_process_msg(char* buf, int buflen)
*\brief		process message received from wpsd.
*
*\param[in]	buf		message buffer.
*\param[in]	buflen	length of message.
*
*\return		OK/ERROR
*/
static int 
_wps_monitor_process_msg(char* buf, int buflen)
{
	WPSD_DATA *pdata;
	WPSD_DATA data;
	pdata = (WPSD_DATA*)buf;
	int ret = OK;

	data.mode = ntohl(pdata->mode);
	data.method = ntohl(pdata->method);
	data.band = ntohl(pdata->band);
	data.action = ntohl(pdata->action);
	data.isApConfig= ntohl(pdata->isApConfig);
	strncpy(data.pinCode, pdata->pinCode, 8);
	
	switch(data.action)
	{
	case WPS_ACT_STOP:
		switch(data.mode)
		{
		case SYS_REPEATER_MODE:
			ret = _wps_repeater_stop(data.band);
			break;
		case SYS_AP_MODE:
			ret = _wps_ap_stop(data.band);
			break;
		default:
			wpsError("mode error");
			ret = ERROR;
		}
		break;
	case WPS_ACT_START:
		switch(data.mode)
		{
		case SYS_REPEATER_MODE:
			if(WPS_UI_METHOD_PBC == data.method)
			{
				ret = _wps_repeater_start(data.band);
			}
			else
			{
				wpsError("method error");
				ret = ERROR;
			}
			break;
		case SYS_AP_MODE:
			if(data.band == WLAN_BAND_2G)
			{
				_wps_backup_UUID_2g(data.band);
			}
			if(WPS_UI_METHOD_PBC == data.method)
			{
				ret = _wps_ap_pbc_start(data.band, data.isApConfig);
			}
			else if(WPS_UI_METHOD_PIN == data.method)
			{
				ret = _wps_ap_pin_start(data.band, data.pinCode, data.isApConfig);
			}
			else
			{
				wpsError("method error");
				ret = ERROR;
			}
			break;
		default:
			wpsError("mode error");
			ret = ERROR;
		}
		break;
	default:
		wpsError("action error");
		ret = ERROR;		
	}

	return ret;
}

/*!
*\fn			 _wps_monitor_mainloop() 
*\brief		 process message what we received in mainloop
*/
void _wps_monitor_mainloop(void)
{
	struct	timeval timeout;
	fd_set	readSet;
	int 	ret;
	int maxSock;
	int nReadSet;
	int band = 0;
	struct sockaddr_in peer;
	socklen_t sa_len;
	int wps_buflen;

	maxSock = wps_ui_sock[0];
	for (band = 1; band < WLAN_BAND_NUM; band++)
	{
		if(maxSock < wps_ui_sock[band])
			maxSock = wps_ui_sock[band];
	}

	while(1)
	{
		if (wps_flag & WPS_FLAG_SHUTDOWN)
		{
			break;
		}

		FD_ZERO(&readSet);
		for (band = 0; band < WLAN_BAND_NUM; band++)
		{
			FD_SET(wps_ui_sock[band], &readSet);
		}

		timeout.tv_sec = 0;
		timeout.tv_usec = 500;	

		nReadSet = select( maxSock + 1, &readSet, NULL, NULL, &timeout);
		if(nReadSet < 0)
		{
			if(errno == EINTR)
			{
				continue;
			}
			wpsError( "select error: %s", strerror(errno));
			goto out;
		}
		else if( nReadSet == 0)/*  time out */
		{
			continue;
		}

		for (band = 0; band < WLAN_BAND_NUM; band++)
		{
			if( FD_ISSET(wps_ui_sock[band], &readSet) )
			{
				bzero(&peer, sizeof(peer));
				sa_len = sizeof(peer);
			    ret = recvfrom(wps_ui_sock[band], wps_readbuf, sizeof(WPSD_DATA), 0, (struct sockaddr*)&peer, &sa_len);
			    if (ret < 0)
			    {
			        wpsError("failed to recvfrom: %s", strerror(errno));
			        goto out;
			    }
				
			    wps_buflen = ret;	
				
				_wps_monitor_process_msg(wps_readbuf, wps_buflen);
			}
		}
	}
	
out: 
	for (band = 0; band < WLAN_BAND_NUM; band++)
	{
		if (wps_ui_sock[band] >= 0)
			close(wps_ui_sock[band]);
		wps_ui_sock[band] = INVALID_SOCKET;
	}
	
	return;
}

/**************************************************************************************************/
/* 									 GLOBAL_FUNCTIONS										   */
/**************************************************************************************************/
/*!
*\fn		 int main(int argc,char *argv[])
*\brief	 main function.
*
*\return	 OK/ERROR
*/
int main(int argc, char* argv[])
{
	int ret = OK;
	FILE* pidfile;

	/* Show usage */
	if (argc > 1) {
		fprintf(stderr, "Usage: wps_monitor\n");
		return ERROR;
	}
	
	_wps_monitor_daemonize();	/* Daemonize */

	/*
	* Check whether this process is running
	*/
	if ((pidfile = fopen(WPS_MONITOR_PID_FILE_PATH, "r"))) 
	{
		wpsError("wps_monitor has been started");

		fclose(pidfile);
		return ERROR;
	}

	/* Create pid file */
	if ((pidfile = fopen(WPS_MONITOR_PID_FILE_PATH, "w"))) {
		fprintf(pidfile, "%d\n", getpid());
		fclose(pidfile);
	}
	else {
		perror("pidfile");
		exit(errno);
	}
	
	/* register signal handler. */
	if (_wps_register_signal() < 0)
	{
		wpsError("wps_monitor register signal handler failed.");
		ret = ERROR;
		goto out;
	}

	if (_wps_monitor_open())
	{
		wpsError("wps_monitor open failed.");
		ret = ERROR;
		goto out;
	}

	_wps_monitor_mainloop();

out:
	/* Destroy pid file */
	unlink(WPS_MONITOR_PID_FILE_PATH);

	return ret;
}
 
